﻿/*
	VERSION:	1.0
	
	
	FUNCTIONS:
		drawMap()							[promise]		Updates the displayed tiles based on provided mapData.  (removes extra layers,  resizes map,  and loads chipset files as needed)
		updateTile()					[null]			Updates a specified tile using the specified tile-data.  (does not create new layers)
		isCompressed()				[t/f]				Checks provided mapData and reports whether or not it is compressed.
		compressMapData()			[compressedMapData]			Receives uncompressed data and returns compressed data suitable for efficient saving.  (can be used as a static function)
		decompressMapData()		[uncompressedMapData]		Receives compressed data and returns decompressed data suitable for editing.  (can be used as a static function)
		
		
	FUNCTION PARAMS
		movieClip = 						makeMap( [instanceName], [target_mc], [depth], [width], [height] )
		null =									updateTile( location, tileData, [tileSize] )
		t/f =										isCompressed( mapData )
		compressedMapData =			compressMapData( uncompressedMapData )
		uncompressedMapData =		decompressMapData( compressedMapData )
		
		
	USAGE:
		#include "functions/map3.as"
		map_mc = makeMap();
		map_mc.drawMap( myMapData );
		
		
		var tileData = mapData.layers[0][0][1] = {
			chipset: 1,
			x: 6,
			y: 0
		}
		var updateLocation = {
			layer: 0,
			x: 0,
			y: 1
		}
		map_mc.updateTile( updateLocation, tileData, newData.tileSize );
		
		
		if( !isCompressed( mapData )  )
		{// if:  data is compressed
			uncompressedMapData = decompressMapData( uncompressedMapData );
			mapData = uncompressedMapData;
		}// if:  data is compressed
		
		
		compressedMapData = compressMapData( uncompressedMapData );
		
		
		decompressed = decompressMapData( compressed );
	
	
	INTERNAL DEPTHS:
		2001		chipsetLoader_mc
*/
function makeMap( instanceName, target_mc, depth, width, height )
{
	////////////////////////////////////////////////////////////////////
	// DEPENDENCIES
	
	#include "addCollisionMethods.as"
	#include "functions/copyObject.as"
	#include "functions/VOW.as"
	#include "functions/VOW/loadBitmap.as"
	#include "functions/nextDepth.as"
	
	
	
	////////////////////////////////////////////////////////////////////
	// ONE-TIME INIT
	
	// static settings
	var layerDepthOffset = 10;
	var encodingRadix = 36;		// 36 is the maximum radix that Flash allows
	
	// create container movieClip
	var target_mc = target_mc || this;
	var depth = (depth != undefined) ? depth : nextDepth(target_mc);
	var instanceName = instanceName || "map"+depth;
	var _this = target_mc.createEmptyMovieClip( instanceName, depth );
	
	// movieClip structure
	_this.tileSize = 16;
	_this.width = width || 20;
	_this.height = height || 15;
	//_this.data = {};
	//_this.data.layers = [];
	//_this.data.collision = [];
	//_this.data.chipsets = [];
	_this.layers_mc_array = [];
	_this.chipset_array = [];
	// _this.onLoad()		// legacy callback after map has been drawn
	
	// legacy support
	_this.collision_array = [];
	addCollisionMethods( _this.collision_array );		// WARNING:  Potential data-leak if external code references these functions directly
	
	// internal shortcuts
	//var data = _this.data;
	var chipsetLoader_mc = _this.createEmptyMovieClip( "chipsetLoader_mc", 2001 );
	chipsetLoader_mc._visible = false;
	var layers_mc_array = _this.layers_mc_array;
	var chipset_array = _this.chipset_array;
	var oldTilesize = _this.tileSize;
	var BitmapData = flash.display.BitmapData;
	var Rectangle = flash.geom.Rectangle;
	var Point = flash.geom.Point;
	
	
	
	////////////////////////////////////////////////////////////////////
	// drawMap()
	
	function drawMap( uncompressedMapData )
	{
		var drawMap_vow = VOW.make();
		if( !isCompressed( uncompressedMapData ))
		{
			uncompressedMapData = decompressMapData( uncompressedMapData );
		}// if:  data is compressed
		var newData = uncompressedMapData;
		
		// break down
		removeExtraLayers();
		resizeExistingLayers();
		// build up
		updateCollision();
		updateProperties();
		addNewLayers();
		loadNewChipsets()
		.then( drawAllTiles, drawMapError )
		.then( finished );
		
		
		function removeExtraLayers()
		{
			// compare known layers against the map-newData
			for(var l=layers_mc_array.length; l>=0; l--)
			{// for:  each existing layer
				var newDataLayer = newData.layers[l];
				if(newDataLayer == undefined)
				{// if:  the newData doesn't have this layer
					// remove this layer from the movieClip
					layers_mc_array[l].pic.dispose();				// delete layer's bitmapData
					layers_mc_array[l].removeMovieClip();		// remove layer's movieClip
					layers_mc_array.splice( l, 1 );					// forget about this layer
				}// if:  the newData doesn't have this layer
			}// for:  each existing layer
		}// removeExtraLayers()
		
		
		function resizeExistingLayers()
		{
			var oldWidth = layers_mc_array[0].pic.width;
			var oldHeight = layers_mc_array[0].pic.height;
			if( (oldWidth != undefined)  &&  (oldTilesize != newData.tileSize  ||  oldWidth != newData.width || oldHeight != newData.height) )
			{// if:  width OR height differ from current bitmaps
				var newWidth = newData.width * newData.tileSize;
				var newHeight = newData.height * newData.tileSize;
				for(var l=layers_mc_array.length; l>=0; l--)
				{// for:  each existing layer
					var thisLayer_mc = layers_mc_array[l];
					thisLayer_mc.pic.dispose();
					thisLayer_mc.pic = new BitmapData( newWidth, newHeight, true, 0 );
					thisLayer_mc.attachBitmap( thisLayer_mc.pic, 0 );
				}// for:  each existing layer
			}// if:  width OR height differ from current bitmaps
			oldTilesize = newData.tileSize;		// remember last-used tileSize for future comparisons
		}// resizeExistingLayers()
		
		
		function updateProperties()
		{
			updateMapData();
			updateCollision();
			
			function updateMapData()
			{
				_this.tileSize = newData.tileSize;
				_this.width = newData.width;
				_this.height = newData.height;
			}// updateMapData()
			
			function updateCollision()
			{
				// clone the collision data
				_this.collision_array.splice(0);		// remove all previous elements  (but keep the array itself to maintain external references)
				copyObject( newData.collision, _this.collision_array );
				addCollisionMethods( _this.collision_array );		// copyObject deletes the previous collision functions
			}// updateCollision()
		}// updateProperties()
		
		
		
		function addNewLayers()
		{
			var newWidth = newData.width * newData.tileSize;
			var newHeight = newData.height * newData.tileSize;
			for(var l=layers_mc_array.length; l<newData.layers.length; l++ ){
				// create a layer
				var newDepth = l * layerDepthOffset;
				var newName = "layer"+l+"_mc";		//->  layer0_mc
				var newLayer_mc = _this.createEmptyMovieClip( newName, newDepth );
				newLayer_mc.pic = new BitmapData( newWidth, newHeight, true, 0 );
				newLayer_mc.attachBitmap( newLayer_mc.pic, 0 );
				layers_mc_array.push( newLayer_mc );
			}// for:  each layer to add
		}// addNewLayers()
		
		
		function loadNewChipsets()
		{
			var loadAllChipsets_vow = VOW.make();
			// compare old chipset newData to new chipset newData
			// // remember all chipsets that already been loaded
			var allLoadedChipsets = {};
			var chipsetsToLoad = [];
			for(var c=0; c<chipset_array.length; c++)
			{// for:  each existing chipset
				allLoadedChipsets[ chipset_array[c].file ] = chipset_array[c].pic;
			}// for:  each existing chipset
			// // make a list of new chipsets that haven't already been loaded
			for(var c=0; c<newData.chipsets.length; c++)
			{// for:  each new chipset
				var newChipName = newData.chipsets[c];
				if(allLoadedChipsets[newChipName] == undefined)
				{// if:  this chipset has NOT been loaded yet
					chipsetsToLoad.push( newChipName );
				}// if:  this chipset has NOT been loaded yet
			}// for:  each new chipset
			// // load the remaining chipsets
			if(chipsetsToLoad.length > 0)
			{// if:  there are more chipsets to load
				var load_array = [];
				for(var l=0; l<chipsetsToLoad.length; l++)
				{// for:  each missing chipset
					// load & keep track of each chipset
					var load_promise = loadChipset( chipsetsToLoad[l] );
					load_array.push( load_promise );
				}// for:  each missing chipset
				// wait for all chipsets to finish loading
				VOW.every( load_array )
				.then( afterAllChipsetsLoad, loadChipsetError );
			}// if:  there are more chipsets to load
			else
			{// if:  no additional chipsets needed
				// arrange chipsets, old and new, into the order specified by the new newData
				arrangeChipsets();
				// report success because no loading is needed
				loadAllChipsets_vow.keep();
			}// if:  no additional chipsets needed
			
			
			function loadChipset( fileName )
			{
				var vow = VOW.make();
				// Load the new image
				var load_promise = loadBitmap( fileName, chipsetLoader_mc );
				load_promise.then( afterLoading, error );
				
				// If successful...
				function afterLoading( newChipset_pic ){
					var chipsetData = {
						file: fileName,
						pic: newChipset_pic
					}
					vow.keep( chipsetData );
				}// afterLoading()
				function error( reason ){
					vow.doBreak( reason );
				}// error()
				
				return vow.promise;
			}// loadChipset()
			
			
			function afterAllChipsetsLoad( allNewChipsets )
			{
				// after all chipsets have loaded
				// add new loaded chipsets to allLoadedChipsets list
				for(var c=0; c<allNewChipsets.length; c++)
				{// for:  each new chipset
					var thisChipset = allNewChipsets[c];
					var newFile = thisChipset.file;
					var newPic = thisChipset.pic;
					allLoadedChipsets[ newFile ] = newPic;
				}// for:  each new chipset
				// arrange chipsets, old and new, into the order specified by the new newData
				arrangeChipsets();
				// report success
				loadAllChipsets_vow.keep();
			}// afterAllChipsetsLoad()
			
			
			function arrangeChipsets()
			{
				// chipset_array  (overall list for displayed map)
				chipset_array.splice(0);		// clear all previous elements
				for(var c=0; c<newData.chipsets.length; c++)
				{// for:  each chipset specified in the new data
					var filePath = newData.chipsets[c];
					var thisChipset = chipset_array[c] = {};
					thisChipset.file = filePath;
					thisChipset.pic = allLoadedChipsets[filePath];
				}// for:  each chipset specified in the new data
			}// arrangeChipsets()
			
			
			function loadChipsetError( reason ){
				loadAllChipsets_vow.doBreak( reason );
			}// loadChipsetError()
			
			
			return loadAllChipsets_vow.promise;
		}// loadNewChipsets()
		
		
		
		
		function drawAllTiles()
		{
			// for:  each tile on each layer
			for(var l=0; l<newData.layers.length; l++)
			{// for:  each layer
				for(var xx=0; xx<newData.width; xx++)
				{// for:  map width
					for(var yy=0; yy<newData.height; yy++)
					{// for:  map height
						var location = {
							layer: l,
							x: xx,
							y: yy
						}
						var tileData = newData.layers[l][xx][yy];
						updateTile( location, tileData, newData.tileSize );
					}// for:  map height
				}// for:  map width
			}// for:  each layer
		}// drawAllTiles()
		
		
		
		function finished()
		{
			drawMap_vow.keep();
		}// finished()
		
		
		
		function drawMapError( reason ){
			drawMap_vow.doBreak( reason );
		}// drawMapError()
		
		
		return drawMap_vow.promise;
	}// drawMap()
	
	
	
	////////////////////////////////////////////////////////////////////
	// updateTile()
	
	function updateTile( location, tileData, tileSize )
	{
		// EXTERNAL DEPENDENCIES:
		// chipset_array
		// layers_mc_array
		// Rectangle
		// Point
		
		var l = location.layer;
		var xx = location.x;
		var yy = location.y;
		//var tileData = newData.layers[l][xx][yy];
		var tileSize = tileSize || _this.tileSize;
		
		// copy from chipset -> bitmap layer
		var srcChipset_pic = chipset_array[ tileData.chipset ].pic;
		var destLayer_pic = layers_mc_array[l].pic;
		var xCopy = tileData.x * tileSize;
		var yCopy = tileData.y * tileSize;
		var copy = new Rectangle( xCopy,yCopy, tileSize,tileSize );
		var paste = new Point( xx * tileSize,  yy * tileSize );
		destLayer_pic.copyPixels( srcChipset_pic, copy, paste );
	}// updateTile()
	
	
	
	////////////////////////////////////////////////////////////////////
	// isCompressed()
	
	function isCompressed( data ){
		if( data.isCompressed === true )		return true;
		if( data.collision[0] !== undefined )		return true;
		return false;
	}// isCompressed()
	
	
	
	////////////////////////////////////////////////////////////////////
	// compressMapData()
	
	function compressMapData( uncompressedData )
	{
		var newData = {};
		
		// copy root variables
		newData.format = uncompressedData.format;
		newData.isCompressed = true;
		newData.tileSize = uncompressedData.tileSize;
		newData.width = uncompressedData.width;
		newData.height = uncompressedData.height;
		
		
		// copy chipsets
		newData.chipsets = [];
		for(var c=0; c<uncompressedData.chipsets.length; c++){
			newData.chipsets[c] = uncompressedData.chipsets[c];
		}// for:  each chipset
		
		
		// compress tiles
		newData.layers = [];
		for(var l=0; l<uncompressedData.layers.length; l++){
			newData.layers[l] = "";
			var thisLayer = uncompressedData.layers[l];
			for(var xx=0; xx<newData.width; xx++){
				for(var yy=0; yy<newData.height; yy++){
					var thisTile = thisLayer[xx][yy];
					var chipset = thisTile.chipset;
					var xChip = thisTile.x;
					var yChip = thisTile.y;
					var compressedValue = compressTileValue( chipset, xChip, yChip );
					newData.layers[l] += compressedValue;
				}// for:  vert tiles
			}// for:  horz tiles
		}// for:  each layer
		// convert layers
		
		
		// compress collision
		newData.collision = "";
		for(var xx=0; xx<newData.width; xx++){
			for(var yy=0; yy<newData.height; yy++){
				var uncompressedValue = Number( uncompressedData.collision[xx][yy] );
				var compressedValue = uncompressedValue.toString( 36 );
				newData.collision += compressedValue;
			}// for:  vert tiles
		}// for:  horz tiles
		
		
		return newData;
		
		
		
		function compressTileValue( chipset, x, y ){
			var chip_code = Number(chipset).toString(36);
			var x_code = Number(x).toString(36);
			var y_code = Number(y).toString(36);
			var output = chip_code + x_code + y_code;
			return output;
		}// compressTileValue()
	}// compressMapData()
	
	
	
	////////////////////////////////////////////////////////////////////
	// decompressMapData()
	
	function decompressMapData( compressedData )
	{
		var newData = {};
		
		// copy root variables
		newData.format = compressedData.format;
		newData.isCompressed = false;
		newData.tileSize = compressedData.tileSize;
		newData.width = compressedData.width;
		newData.height = compressedData.height;
		
		
		// copy chipsets
		newData.chipsets = [];
		for(var c=0; c<compressedData.chipsets.length; c++){
			newData.chipsets[c] = compressedData.chipsets[c];
		}// for:  each chipset
		
		
		// de-compress tiles
		newData.layers = [];
		var readIndex = 0;
		var readLength = 3;
		for(var l=0; l<compressedData.layers.length; l++){
			// read data
			readIndex = 0;
			var layer_str = compressedData.layers[l];
			// write data
			newData.layers[l] = [];
			for(var xx=0; xx<newData.width; xx++){
				newData.layers[l][xx] = [];
				for(var yy=0; yy<newData.height; yy++){
					// read data
					var tile_str = layer_str.substr( readIndex, readLength );
					readIndex += readLength;
					// write data
					var thisTile = newData.layers[l][xx][yy] = {};
					thisTile.chipset = decompressValue( tile_str.charAt(0) );
					thisTile.x = decompressValue( tile_str.charAt(1) );
					thisTile.y = decompressValue( tile_str.charAt(2) );
				}// for:  each layer
			}// for:  each layer
		}// for:  each layer
		// convert layers
		
		
		// de-compress collision
		newData.collision = [];
		var readIndex = 0;
		var readLength = 1;
		for(var xx=0; xx<newData.width; xx++){
			newData.collision[xx] = [];
			for(var yy=0; yy<newData.height; yy++){
				// read value
				var compressedValue = compressedData.collision.substr( readIndex, readLength );
				var uncompressedValue = decompressValue( compressedValue );
				readIndex += readLength;
				// write value
				newData.collision[xx][yy] = uncompressedValue;
			}// height
		}// width
		
		
		return newData;
		
		
		
		function decompressValue( encodedValue ){
			if( encodedValue=="0x" ){
				return 33;
			}else{
				return parseInt( encodedValue, 36 );
			}
		}// decompressValue()
	}// decompressMapData()
	
	
	
	////////////////////////////////////////////////////////////////////
	// INTERFACE
	
	_this.drawMap = drawMap;
	_this.updateTile = updateTile;
	_this.isCompressed = isCompressed;
	_this.compressMapData = compressMapData;
	_this.decompressMapData = decompressMapData;
	
	return _this;
}// makeMap()